/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  DMA.c
  \brief	  This is the DMA Driver File
  \brief	  Configures the DMA HW providing several instances.
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 139 $
  \date  	  $Date: 2011-01-27 16:22:55 -0600 (Thu, 27 Jan 2011) $
  
  * History:  20/August/2008 - Initial Version (IM)
  *   		  12/May/2009 - MISRA Check, Release (IM)
  			  17/May/2010 - Add new functions, deprecating others. A new smaller structure is required
  			  				for future upgrades and reduced memory usage (IM)
  			  15/Aug/2010 - Added callback functions for context switching. Optimized code on interrupt functions (IM)
  			  27/Jan/2011 - Added request support: not only Always request but other peripherals request				

  * MISRA VIOLATIONS:
	- [ MISRA 11.2 ]
	- [ MISRA 16.9 ]

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/

#include	 "DMA_drv.h"

/* Data Types  */

#define  	DMA_INITIALIZED  	  (1u)
#define  	DMA_UNINITIALIZED	  (0u)

/* Driver Variables */
uint8_t	DMA_ChannelStatus[ DMA_MAX_DMACHANNELS ];
uint8_t	DMA_RequestEnabled[ DMA_MAX_DMACHANNELS ];
uint8_t	DMA_DriverStatus = DMA_UNINITIALIZED;

// temp vars for profiling OK to remove if you don't use them
//extern uint32_t timing[];
//extern uint8_t DMAinc;

DMA_CallbackType DMA_Callback[ DMA_MAX_DMACHANNELS ];

volatile struct EDMA_tag	* EDMA_PTR = &EDMA;
//compare against

#ifdef DCU_LITE
DMA_ContextCallbackType DMA_CtxDCU;
DMA_ContextCallbackType DMA_CtxDCULITE;
#endif

uint8_t channelAdjust[32];

void DMA_RegisterTest(void)
{
  
}

uint32_t endianness_correction_8bit(uint32_t array_index)
{
    int correct_array_index, difference =0;
    difference = array_index%4;
    switch(difference)
    { 
        case(0):
	    correct_array_index = array_index + 3;
            break;
	case(1):
	    correct_array_index = array_index + 1;
            break;
	case(2):
	    correct_array_index = array_index - 1;
            break;
	case(3):
	    correct_array_index = array_index - 3;
            break;
    }
    return correct_array_index;

}

/**
* \brief	DMA_Init - Initializes the DMA driver control variables
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void DMA_Init( void )
{
	uint16_t i;
        
        EDMA_PTR = &EDMA;
        for(i = 0u; i<DMA_MAX_DMACHANNELS; i+=4)
        {
         channelAdjust[i] = i+3;
         channelAdjust[i+1] = i+2;
         channelAdjust[i+2] = i+1;
         channelAdjust[i+3] = i;        
        }
	DMA_UnInit( );

	for( i = 0u; i < DMA_MAX_DMACHANNELS; i++ )
	{
		DMA_ChannelStatus[ i ]	= DMA_CH_IDLE;
		DMA_RequestEnabled[ i ] = 0u;
		/* VIOLATION TO THE FOLLOWING RULES */
		/* [ MISRA 11.2 ] The value of an expression shall not be impicitly converted to a different underlying type if:... */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		/* the null pointer value indicates the function has not been initialized */
		DMA_Callback[ i ]		= NULL_PTR;
	}

#ifdef DCU_LITE
	DMA_CtxDCU = NULL_PTR;
	DMA_CtxDCULITE = NULL_PTR;
#endif

	/*	Enhanced mode ON */
	EDMA.CR.B.EMLM	 = 1u;
	DMA_DriverStatus = DMA_INITIALIZED;	
}

#ifdef DCU_LITE
void DMA_SetDCUContextCallback(DMA_ContextCallbackType callback)
{
	DMA_CtxDCU = callback;
}

void DMA_SetDCULITEContextCallback(DMA_ContextCallbackType callback)
{
	DMA_CtxDCULITE = callback;
}
#endif

/**
* \brief	DMA_UnInit - Deactivates the DMA HW and driver
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void DMA_UnInit( void )
{
	EDMA.ERQL.R	 = 0u;
	EDMA.CR.R		 = 0x00000000u;
        EDMA.CR.B.GRP1PRI = 1u;
	EDMA.INTL.R	 = 0u;
	EDMA.EEIL.R	 = 0u;
	EDMA.ERRL.R		 = 0u;
	DMA_DriverStatus = DMA_UNINITIALIZED;
}

/**
* \brief	DMA_InitChannel - Initializes a single channel. Sets this channel
* \brief	as used and cannot be initialized by different entity. It inits
* \brief	the channel HW with its basic functionality.
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be initialized.
* \return	DMA_ErrorType, possible resutls: DMA_ERROR_UNINIT, DMA_ERROR_RANGE, DMA_ERROR_BUSY, DMA_ERROR_OK
* \todo
*/
DMA_ErrorType DMA_InitChannel( uint8_t Channel )
{
	DMA_ErrorType error;
	uint32_t *dst;
	uint32_t i;

	error = DMA_ERROR_OK;

	if( DMA_DriverStatus == DMA_UNINITIALIZED )
	{
		error = DMA_ERROR_UNINIT;
	}
	else  if( Channel >= DMA_MAX_DMACHANNELS )
	{
		error = DMA_ERROR_RANGE;
	}
	else  if( DMA_ChannelStatus[ Channel ] != DMA_CH_IDLE )
	{
		error = DMA_ERROR_BUSY;
	}
	else
 
	{
		/*  Set the 8 words of TCD = 0 */
		dst = (uint32_t *)(&EDMA.TCD[Channel].SADDR);
		i = 8;
		while(i)
		{
			*dst = 0;
			dst++;
			i--;
		}
		//EDMA.TCD[ Channel ].D_REQ		= 1u;
		//EDMA.TCD[ Channel ].BWC			= 2u;
		DMA_ChannelStatus[ Channel ]	= DMA_CH_PREPARED;
		DMA_RequestEnabled[ Channel ]	= 0u;
		EDMA.CINT.R						= Channel;
		
                EDMA.CPR[channelAdjust[Channel]].R			= Channel;

	}
return error;
}

/**
* \brief	DMA_UnInitChannel - Uninitializes a single channel. Deallocates the
* \brief	channel and turns off the channel (interrupts)
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be uninitialized.
* \return	void
* \todo
*/
void DMA_UnInitChannel( uint8_t Channel )
{
	uint32_t *dst;
	uint32_t i;
	
	EDMA.CDNE.R						= Channel;
	DMA_ChannelStatus[ Channel ]	= DMA_CH_IDLE;
	DMA_RequestEnabled[ Channel ]	= 0u;
	EDMA.CINT.R						= Channel;
	EDMA.CERQ.R						= Channel;
	DMAMUXx.CHCONFIG[endianness_correction_8bit(Channel)].B.SOURCE = 0;
	DMAMUXx.CHCONFIG[endianness_correction_8bit(Channel)].B.ENBL	= 0;	

	/*  Set the 8 words of TCD = 0 */
	dst = (uint32_t *)(&EDMA.TCD[Channel].SADDR);
	i = 8;
	while(i)
	{
		*dst = 0;
		dst++;
		i--;
	}
}


/**
* \brief	DMA_SetCallback - Turns on/off the eDMA channel interrupts and sets 
* \brief	the callback function for the interrupt. Use NULL_PTR to turn off
* \brief	interrupts and callbacks.
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be configured
* \param	DMA_CallbackType Callback, it is the callback function to be used.
* \return	DMA_ErrorType, possible resutls: DMA_ERROR_UNINIT, DMA_ERROR_RANGE, DMA_ERROR_CHANNEL_UNINIT, DMA_ERROR_OK
* \todo
*/
DMA_ErrorType DMA_SetCallback( uint8_t Channel, DMA_CallbackType Callback )
{
	DMA_ErrorType error;

	error = DMA_ERROR_OK;

	if( DMA_DriverStatus == DMA_UNINITIALIZED )
	{
		error = DMA_ERROR_UNINIT;
	}
	else  if( Channel >= DMA_MAX_DMACHANNELS )
	{
		error = DMA_ERROR_RANGE;
	}
	else  if( DMA_ChannelStatus[ Channel ] == DMA_CH_IDLE )
	{
		error = DMA_ERROR_CHANNEL_UNINIT;
	}
	else

	{
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */
		DMA_Callback[ Channel ] = Callback;
		if( Callback != NULL_PTR )
		{
			EDMA.CINT.R				= Channel;
			EDMA.TCD[ Channel ].INT_MAJ = 1u;
		}
		else
		{
			EDMA.TCD[ Channel ].INT_MAJ = 0u;
			EDMA.CINT.R				= Channel;
		}
		error = DMA_ERROR_OK;
	}
	return error;
}



/**
* \brief	DMA_EnableRequest - Configures the eDMA request for the specific
* \brief	channel. For example, the SPI can request service from the eDMA.
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be configured.
* \param	uint8_t Source, it is the source that will request service from the
* \param	eDMA channel. 
* \return	DMA_ErrorType, possible resutls: DMA_ERROR_UNINIT, DMA_ERROR_RANGE, DMA_ERROR_CHANNEL_UNINIT, DMA_ERROR_OK
* \todo
*/
DMA_ErrorType DMA_EnableRequest( uint8_t Channel, uint8_t Source )
{
	DMA_ErrorType error;

	error = DMA_ERROR_OK;

	if( DMA_DriverStatus == DMA_UNINITIALIZED )
	{
		error = DMA_ERROR_UNINIT;
	}
	else  if( Channel >= DMA_MAX_DMACHANNELS )
	{
		error = DMA_ERROR_RANGE;
	}
	else  if( DMA_ChannelStatus[ Channel ] == DMA_CH_IDLE )
	{
		error = DMA_ERROR_CHANNEL_UNINIT;
	}
	else
	{
		DMAMUXx.CHCONFIG[endianness_correction_8bit(Channel)].B.SOURCE = Source;
		DMAMUXx.CHCONFIG[endianness_correction_8bit(Channel)].B.ENBL	= 1u;
		if( (Source >= DMA_REQ_ALWAYS_START) && (Source <= DMA_REQ_ALWAYS_END) )
		{
			/* EDMA.SERQR.R = Channel; */
			/* This register will be set later, in DMA_Start(x) */
			EDMA.TCD[ Channel ].D_REQ		= 1u;
			DMA_RequestEnabled[ Channel ]	= 1u;				
		}
		else
		{
			EDMA.SERQ.R = Channel;			
		}
		
	}
	return error;
}

#ifndef REMOVE_DEPRECATED
/** DEPRECATED-DEPRECATED-DEPRECATED-DEPRECATED-DEPRECATED-DEPRECATED-DEPRECATED-DEPRECATED-DEPRECATED 
* \brief	DMA_Configure - Configures a single channel with the parameters
* \brief	specified on the config structure.
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be configured.
* \param	DMA_ChannelCfgType Configuration, set of parameters to be used on the channel.
* \param	eDMA channel. 
* \return	DMA_ErrorType, possible resutls: DMA_ERROR_UNINIT, DMA_ERROR_RANGE, DMA_ERROR_CHANNEL_UNINIT, DMA_ERROR_OK
* \todo
*/
DMA_ErrorType DMA_Configure( uint8_t Channel, DMA_ChannelCfgType *Configuration )
{
	DMA_ErrorType error;

	error = DMA_ERROR_OK;

	if( DMA_DriverStatus == DMA_UNINITIALIZED )
	{
		error = DMA_ERROR_UNINIT;
	}
	else  if( Channel >= DMA_MAX_DMACHANNELS )
	{
		error = DMA_ERROR_RANGE;
	}
	else  if( DMA_ChannelStatus[ Channel ] == DMA_CH_IDLE )
	{
		error = DMA_ERROR_CHANNEL_UNINIT;
	}
	else

	{

		EDMA.TCD[ Channel ].SADDR	= Configuration->SourceAddress;
		EDMA.TCD[ Channel ].DADDR	= Configuration->DestinationAddress;
		EDMA.TCD[ Channel ].SOFF	= Configuration->SourceOffset;
		EDMA.TCD[ Channel ].DOFF 	= Configuration->DestinationOffset;
		EDMA.TCD[ Channel ].BITER	= Configuration->MajorLoopCount;
		EDMA.TCD[ Channel ].CITER	= Configuration->MajorLoopCount;
		EDMA.TCD[ Channel ].SLAST 	= Configuration->LastSourceAdjustment;
		EDMA.TCD[ Channel ].DLAST_SGA = Configuration->LastDestinationAdjustment;
		EDMA.TCD[ Channel ].NBYTESu.R	= 0u;
		EDMA.TCD[ Channel ].NBYTESu.B.SMLOE = Configuration->MinorLoopOffsetSrc;
		EDMA.TCD[ Channel ].NBYTESu.B.DMLOE = Configuration->MinorLoopOffsetDest;
		EDMA.TCD[ Channel ].NBYTESu.B.MLOFF  = Configuration->MinorLoopOffset;
		EDMA.TCD[ Channel ].NBYTESu.R		|= Configuration->MinorLoopCount;

		error = DMA_ERROR_OK;
	}

return error;
}
#endif


/**
* \brief	DMA_Start - Triggers the start of the DMA by software, if the
* \brief	DMA is not busy.
* \author	void
* \return	void
* \todo
*/
DMA_ErrorType DMA_Start( uint8_t Channel )
{

	DMA_ErrorType error;
	error = DMA_ERROR_OK;

	if( EDMA.TCD[ Channel ].ACTIVE != 0u )
	{
		error = DMA_ERROR_BUSY;
	}
	else  if( DMA_ChannelStatus[ Channel ] == DMA_CH_IDLE )
	{
		error = DMA_ERROR_CHANNEL_UNINIT;
	}
	else
	{
#ifdef DCULITE
		if(DMA_ChannelStatus[ Channel ] == DMA_CH_PREPARED)
		{
			if(DCU_GetCurrentDCU() == DCU_DCU)
			{
				DMA_ChannelStatus[ Channel ] = DMA_CH_RUNNING;	
			}
			else
			{
				DMA_ChannelStatus[ Channel ] = DMA_CH_RUNNING_LITE;
			}
		}		
		else if(DMA_ChannelStatus[ Channel ] == DMA_CH_READY ||
		   DMA_ChannelStatus[ Channel ] == DMA_CH_RUNNING ||
		   DMA_ChannelStatus[ Channel ] == DMA_CH_READY_LSBFVS)
		{
			DMA_ChannelStatus[ Channel ] = DMA_CH_RUNNING;	
		}		
		else
		{
			DMA_ChannelStatus[ Channel ] = DMA_CH_RUNNING_LITE;	
		}
#else
		DMA_ChannelStatus[ Channel ] = DMA_CH_RUNNING;	
#endif	
		if( DMA_RequestEnabled[ Channel ] == 1u )
		{
			DMA_RequestEnabled[ Channel ] = 0u;
			EDMA.SERQ.R	= Channel;
			EDMA.SSRT.R = Channel;
		}
		else
		{
			EDMA.SSRT.R = Channel;
		}
	}

	return error;
}

/**
* \brief	DMA_SetReady - Marks the channel, as a channel that can be triggered
* \brief	to start a trasnfer.
* \author	void
* \return	void
* \todo
*/
DMA_ErrorType DMA_SetReady( uint8_t Channel )
{

	DMA_ErrorType error;

	if( DMA_ChannelStatus[ Channel ] == DMA_CH_IDLE )
	{
		error = DMA_ERROR_CHANNEL_UNINIT;
	}
	else
	{
#ifdef DCULITE	
		if(DCU_GetCurrentDCU() == DCU_DCU)
		{
			DMA_ChannelStatus[ Channel ] = DMA_CH_READY;	
		}		
		else
		{
			DMA_ChannelStatus[ Channel ] = DMA_CH_READY_LITE;	
		}
#else
		DMA_ChannelStatus[ Channel ] = DMA_CH_READY;
#endif	
		error = DMA_ERROR_OK;
	}

	return error;
}

/**
* \brief	DMA_SetReady_LSBFVS - Marks the channel, as a channel that can be triggered (on LSBFVS)
* \brief	to start a trasnfer.
* \author	void
* \return	void
* \todo
*/
DMA_ErrorType DMA_SetReady_LSBFVS( uint8_t Channel )
{

	DMA_ErrorType error;

	if( DMA_ChannelStatus[ Channel ] == DMA_CH_IDLE )
	{
		error = DMA_ERROR_CHANNEL_UNINIT;
	}
	else
	{
#ifdef DCULITE	
		if(DCU_GetCurrentDCU() == DCU_DCU)
		{
			DMA_ChannelStatus[ Channel ] = DMA_CH_READY_LSBFVS;	
		}		
		else
		{
			DMA_ChannelStatus[ Channel ] = DMA_CH_READY_LSBFVS_LITE;	
		}
#else
		DMA_ChannelStatus[ Channel ] = DMA_CH_READY_LSBFVS;
#endif	
		error = DMA_ERROR_OK;
	}

	return error;
}

/**
* \brief	DMA_Lock - Locks the DMA, it must be used when the channel is performing
* \brief	a trasnfer.
* \author	void
* \return	void
* \todo
*/
DMA_ErrorType DMA_Lock( void )
{

	return DMA_ERROR_OK;
}


/* This private function will perform the DMA callback */
/* in case of double display systems it will correctly switch context, depending the current one */
static void DMA_ISR(uint8_t channel)
{
#ifdef DCULITE
	uint8_t ctx;
	ctx = DCU_GetCurrentDCU();	
#endif		

	EDMA.CDNE.R	= channel;
	EDMA.CERQ.R	= channel;
	EDMA.CINT.R	= channel;
	DMAMUXx.CHCONFIG[endianness_correction_8bit(channel)].B.ENBL	= 0;	
	
#ifdef DCULITE		
	if(DMA_ChannelStatus[channel] == DMA_CH_RUNNING)
	{
		if(DMA_CtxDCU != NULL_PTR)
			DMA_CtxDCU();
	}	
	else
	{
		if(DMA_CtxDCULITE != NULL_PTR)
			DMA_CtxDCULITE();
	}
#endif
	/*Call back, if exists */
	if(DMA_Callback[channel] != NULL_PTR)	
		DMA_Callback[channel](channel);

#ifdef DCULITE	
	if(ctx == DCU_DCU)
	{
		if(DMA_CtxDCU != NULL_PTR)
			DMA_CtxDCU();
	}	
	else
	{
		if(DMA_CtxDCULITE != NULL_PTR)
			DMA_CtxDCULITE();
	}
#endif	
}

/**
* \brief	DMA_CH0_ISR - DMA Channel ISR 0
* \author	void
* \return	void
* \todo
*/

void DMA0_ISR_Vybrid(){
  uint32_t i,x = 1;
  //For performance profiling, does not belong here
  //timing[DMAinc] = (*(uint32_t *)(PIT_CVAL2));
  //DMAinc++;
  for(i=0;i<32;i++){
    if (EDMA.INTL.R & x) DMA_ISR(i);
    x = x << 1;
  }
  
}
void DMA_CH0_ISR( void )
{
	DMA_ISR(0);	
}

/**
* \brief	DMA_CH1_ISR - DMA Channel ISR 1
* \author	void
* \return	void
* \todo
*/
void DMA_CH1_ISR( void )
{
	DMA_ISR(1);	
}

/**
* \brief	DMA_CH2_ISR - DMA Channel ISR 2
* \author	void
* \return	void
* \todo
*/
void DMA_CH2_ISR( void )
{
	DMA_ISR(2);	
}

/**
* \brief	DMA_CH3_ISR - DMA Channel ISR 3
* \author	void
* \return	void
* \todo
*/
void DMA_CH3_ISR( void )
{
	DMA_ISR(3);	
}

/**
* \brief	DMA_CH4_ISR - DMA Channel ISR 4
* \author	void
* \return	void
* \todo
*/
void DMA_CH4_ISR( void )
{
	DMA_ISR(4);	
}

/**
* \brief	DMA_CH5_ISR - DMA Channel ISR 5
* \author	void
* \return	void
* \todo
*/
void DMA_CH5_ISR( void )
{
	DMA_ISR(5);	
}

/**
* \brief	DMA_CH6_ISR - DMA Channel ISR 6
* \author	void
* \return	void
* \todo
*/
void DMA_CH6_ISR( void )
{
	DMA_ISR(6);	
}

/**
* \brief	DMA_CH7_ISR - DMA Channel ISR 7
* \author	void
* \return	void
* \todo
*/
void DMA_CH7_ISR( void )
{
	DMA_ISR(7);	
}

/**
* \brief	DMA_CH8_ISR - DMA Channel ISR 8
* \author	void
* \return	void
* \todo
*/
void DMA_CH8_ISR( void )
{
	DMA_ISR(8);	
}

/**
* \brief	DMA_CH9_ISR - DMA Channel ISR 9
* \author	void
* \return	void
* \todo
*/
void DMA_CH9_ISR( void )
{
	DMA_ISR(9);	
}

/**
* \brief	DMA_CH10_ISR - DMA Channel ISR 10
* \author	void
* \return	void
* \todo
*/
void DMA_CH10_ISR( void )
{
	DMA_ISR(10);	
}

/**
* \brief	DMA_CH11_ISR - DMA Channel ISR 11
* \author	void
* \return	void
* \todo
*/
void DMA_CH11_ISR( void )
{
	DMA_ISR(11);	
}

/**
* \brief	DMA_CH112_ISR - DMA Channel ISR 11
* \author	void
* \return	void
* \todo
*/
void DMA_CH12_ISR( void )
{
	DMA_ISR(12);	
}

/**
* \brief	DMA_CH13_ISR - DMA Channel ISR 13
* \author	void
* \return	void
* \todo
*/
void DMA_CH13_ISR( void )
{
	DMA_ISR(13);	
}

/**
* \brief	DMA_CH14_ISR - DMA Channel ISR 14
* \author	void
* \return	void
* \todo
*/
void DMA_CH14_ISR( void )
{
	DMA_ISR(14);	
}

/**
* \brief	DMA_CH15_ISR - DMA Channel ISR 15
* \author	void
* \return	void
* \todo
*/
void DMA_CH15_ISR( void )
{
	DMA_ISR(15);	
}
